<?php
/**
 * admin_api.php
 * ----------------------------------------
 * Token management API for KAS Chat system.
 * Supports: create / list / revoke / activate / delete actions
 * Data stored in tokens.json (keyed by token value)
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$tokensFile = 'tokens.json';

/* ------------------- Utilities ------------------- */

function generateToken(): string {
    return bin2hex(random_bytes(16));
}

function loadTokens(): array {
    global $tokensFile;
    if (!file_exists($tokensFile)) {
        file_put_contents($tokensFile, json_encode([], JSON_PRETTY_PRINT));
        return [];
    }
    $json = file_get_contents($tokensFile);
    $data = json_decode($json, true);
    return is_array($data) ? $data : [];
}

function saveTokens(array $tokens): void {
    global $tokensFile;
    file_put_contents($tokensFile, json_encode($tokens, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
}

/* ------------------- Main Logic ------------------- */

$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';

switch ($action) {

    /* CREATE new token */
    case 'create':
        $userName = trim($input['userName'] ?? '');
        $expiryDays = isset($input['expiryDays']) ? intval($input['expiryDays']) : null;

        if ($userName === '') {
            echo json_encode(['success' => false, 'message' => 'User name required']);
            exit;
        }

        $tokens = loadTokens();
        $token = generateToken();

        $tokenData = [
            'user' => $userName,
            'created' => date('Y-m-d H:i:s'),
            'active' => true,
            'last_used' => null,
            'fingerprint' => null
        ];

        if ($expiryDays && $expiryDays > 0) {
            $tokenData['expires'] = date('Y-m-d H:i:s', strtotime("+$expiryDays days"));
        } else {
            $tokenData['expires'] = null;
        }

        $tokens[$token] = $tokenData;
        saveTokens($tokens);

        echo json_encode([
            'success' => true,
            'token' => $token,
            'data' => $tokenData
        ]);
        break;

    /* LIST all tokens */
    case 'list':
        $tokens = loadTokens();
        echo json_encode(['success' => true, 'tokens' => $tokens]);
        break;

    /* REVOKE (disable) a token */
    case 'revoke':
        $token = $input['token'] ?? '';
        if ($token === '') {
            echo json_encode(['success' => false, 'message' => 'Token required']);
            exit;
        }
        $tokens = loadTokens();
        if (isset($tokens[$token])) {
            $tokens[$token]['active'] = false;
            saveTokens($tokens);
            echo json_encode(['success' => true, 'message' => 'Token revoked']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Token not found']);
        }
        break;

    /* ACTIVATE a revoked token */
    case 'activate':
        $token = $input['token'] ?? '';
        if ($token === '') {
            echo json_encode(['success' => false, 'message' => 'Token required']);
            exit;
        }
        $tokens = loadTokens();
        if (isset($tokens[$token])) {
            $tokens[$token]['active'] = true;
            saveTokens($tokens);
            echo json_encode(['success' => true, 'message' => 'Token activated']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Token not found']);
        }
        break;

    /* DELETE a token */
    case 'delete':
        $token = $input['token'] ?? '';
        if ($token === '') {
            echo json_encode(['success' => false, 'message' => 'Token required']);
            exit;
        }
        $tokens = loadTokens();
        if (isset($tokens[$token])) {
            unset($tokens[$token]);
            saveTokens($tokens);
            echo json_encode(['success' => true, 'message' => 'Token deleted']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Token not found']);
        }
        break;

    /* DEFAULT (invalid action) */
    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
        break;
}
