<?php
// ===== HTTP Basic Auth =====
$USERNAME = 'Kaya';                  // your desired username
$PASSWORD = 'Black!59';   // your desired password

if (!isset($_SERVER['PHP_AUTH_USER']) || 
    $_SERVER['PHP_AUTH_USER'] !== $USERNAME || 
    $_SERVER['PHP_AUTH_PW'] !== $PASSWORD) {
    
    header('WWW-Authenticate: Basic realm="Restricted Admin Panel"');
    header('HTTP/1.0 401 Unauthorized');
    echo 'Access denied.';
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Token Management - Admin Panel</title>
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600&display=swap');
    
    * { box-sizing: border-box; }
    
    body {
      font-family: 'Inter', sans-serif;
      margin: 0;
      padding: 20px;
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      min-height: 100vh;
    }
    
    .container {
      max-width: 1200px;
      margin: 0 auto;
      background: white;
      border-radius: 16px;
      padding: 30px;
      box-shadow: 0 20px 60px rgba(0,0,0,0.3);
    }
    
    h1 {
      color: #1e293b;
      margin-top: 0;
      border-bottom: 3px solid #667eea;
      padding-bottom: 15px;
    }
    
    .version-badge {
      display: inline-block;
      background: #10b981;
      color: white;
      padding: 4px 12px;
      border-radius: 12px;
      font-size: 12px;
      font-weight: 600;
      margin-left: 10px;
    }
    
    .section {
      margin: 30px 0;
      padding: 20px;
      background: #f8fafc;
      border-radius: 12px;
      border: 1px solid #e2e8f0;
    }
    
    h2 {
      color: #475569;
      margin-top: 0;
    }
    
    .form-group {
      margin: 15px 0;
    }
    
    label {
      display: block;
      font-weight: 500;
      color: #334155;
      margin-bottom: 5px;
    }
    
    input, select {
      width: 100%;
      padding: 12px;
      border: 1px solid #cbd5e1;
      border-radius: 8px;
      font-family: inherit;
      font-size: 14px;
    }
    
    input:focus, select:focus {
      outline: none;
      border-color: #667eea;
      box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }
    
    button {
      background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
      color: white;
      border: none;
      padding: 12px 24px;
      border-radius: 8px;
      cursor: pointer;
      font-weight: 500;
      font-size: 14px;
      transition: transform 0.2s, box-shadow 0.2s;
    }
    
    button:hover {
      transform: translateY(-2px);
      box-shadow: 0 8px 20px rgba(102, 126, 234, 0.4);
    }
    
    .btn-danger {
      background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    }
    
    .btn-success {
      background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    }
    
    table {
      width: 100%;
      border-collapse: collapse;
      margin-top: 20px;
      background: white;
      border-radius: 8px;
      overflow: hidden;
    }
    
    th, td {
      padding: 12px;
      text-align: left;
      border-bottom: 1px solid #e2e8f0;
    }
    
    th {
      background: #f1f5f9;
      font-weight: 600;
      color: #475569;
    }
    
    tr:hover {
      background: #f8fafc;
    }
    
    .token-cell {
      font-family: monospace;
      font-size: 12px;
      word-break: break-all;
    }
    
    .status-active {
      color: #10b981;
      font-weight: 500;
    }
    
    .status-revoked {
      color: #ef4444;
      font-weight: 500;
    }
    
    .actions {
      display: flex;
      gap: 8px;
    }
    
    .actions button {
      padding: 6px 12px;
      font-size: 12px;
    }
    
    .copy-btn {
      cursor: pointer;
      color: #667eea;
      text-decoration: underline;
      font-size: 12px;
    }
    
    .copy-btn:hover {
      color: #764ba2;
    }
    
    .alert {
      padding: 15px;
      border-radius: 8px;
      margin: 15px 0;
    }
    
    .alert-success {
      background: #d1fae5;
      color: #065f46;
      border: 1px solid #10b981;
    }
    
    .alert-error {
      background: #fee2e2;
      color: #991b1b;
      border: 1px solid #ef4444;
    }
    
    .version-info {
      background: #eff6ff;
      border: 1px solid #3b82f6;
      padding: 12px;
      border-radius: 8px;
      margin-bottom: 20px;
      font-size: 13px;
      color: #1e40af;
    }
  </style>
</head>
<body>
  <div class="container">
    <h1>🔐 Token Management System <span class="version-badge" id="versionBadge">v2.0.0</span></h1>
    
    <div class="version-info">
      <strong>⚠️ Important:</strong> Current chatbot version is <strong id="displayVersion">2.0.0</strong>. 
      All new share URLs will include this version to ensure users get the latest updates.
      <br><strong>Remember:</strong> Update the APP_VERSION in both index.html and this admin panel when you make changes!
    </div>
    
    <div class="section">
      <h2>Create New Token</h2>
      <div class="form-group">
        <label>User Name</label>
        <input type="text" id="userName" placeholder="Enter user name">
      </div>
      <div class="form-group">
        <label>Expiry (optional)</label>
        <select id="expiryDays">
          <option value="">Never expires</option>
          <option value="7">7 days</option>
          <option value="30">30 days</option>
          <option value="90">90 days</option>
          <option value="365">365 days</option>
        </select>
      </div>
      <button onclick="createToken()">Create Token</button>
      <div id="createResult"></div>
    </div>
    
    <div class="section">
      <h2>Active Tokens</h2>
      <button onclick="loadTokens()">Refresh List</button>
      <div id="tokensList"></div>
    </div>
  </div>

  <script>
    const ADMIN_API = 'admin_api.php';
    
    // ===== VERSION CONTROL - MUST MATCH index.html =====
    const APP_VERSION = '2.0.0';
    // ===================================================
    
    const BASE_URL = window.location.origin + window.location.pathname.replace('admin.html', 'index.html').replace('admin_panel.php', 'index.html');
    
    // Display version on page load
    document.getElementById('versionBadge').textContent = 'v' + APP_VERSION;
    document.getElementById('displayVersion').textContent = APP_VERSION;
    
    async function createToken() {
      const userName = document.getElementById('userName').value.trim();
      const expiryDays = document.getElementById('expiryDays').value;
      
      if (!userName) {
        showResult('createResult', 'Please enter a user name', 'error');
        return;
      }
      
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            action: 'create',
            userName: userName,
            expiryDays: expiryDays ? parseInt(expiryDays) : null
          })
        });
        
        const data = await response.json();
        
        if (data.success) {
          const shareUrl = BASE_URL + '?token=' + data.token + '&v=' + APP_VERSION;
          const message = `
            <strong>Token created successfully!</strong><br>
            <strong>User:</strong> ${data.data.user}<br>
            <strong>Token:</strong> <span class="token-cell">${data.token}</span><br>
            <strong>Share URL:</strong> <span class="copy-btn" onclick="copyToClipboard('${shareUrl}')">${shareUrl}</span><br>
            ${data.data.expires ? '<strong>Expires:</strong> ' + data.data.expires : '<strong>Never expires</strong>'}
          `;
          showResult('createResult', message, 'success');
          document.getElementById('userName').value = '';
          loadTokens();
        } else {
          showResult('createResult', 'Error: ' + data.message, 'error');
        }
      } catch (error) {
        showResult('createResult', 'Error creating token: ' + error.message, 'error');
      }
    }
    
    async function loadTokens() {
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ action: 'list' })
        });
        
        const data = await response.json();
        
        if (data.success) {
          displayTokens(data.tokens);
        }
      } catch (error) {
        console.error('Error loading tokens:', error);
      }
    }
    
    function displayTokens(tokens) {
      const container = document.getElementById('tokensList');
      
      if (Object.keys(tokens).length === 0) {
        container.innerHTML = '<p>No tokens found.</p>';
        return;
      }
      
      let html = '<table><thead><tr><th>User</th><th>Token</th><th>Status</th><th>Device Bound</th><th>Created</th><th>First Used</th><th>Last Used</th><th>Access Count</th><th>Unauthorized Attempts</th><th>Actions</th></tr></thead><tbody>';
      
      for (const [token, info] of Object.entries(tokens)) {
        const shareUrl = BASE_URL + '?token=' + token + '&v=' + APP_VERSION;
        const statusClass = info.active ? 'status-active' : 'status-revoked';
        const statusText = info.active ? 'Active' : 'Revoked';
        const deviceBound = info.bound_fingerprint ? '✓ Yes' : '✗ Not Yet';
        const deviceClass = info.bound_fingerprint ? 'status-active' : 'status-revoked';
        const accessCount = info.access_count || 0;
        const unauthorizedCount = info.unauthorized_attempts ? info.unauthorized_attempts.length : 0;
        const unauthorizedClass = unauthorizedCount > 0 ? 'status-revoked' : '';
        
        html += `
          <tr>
            <td>${info.user}</td>
            <td class="token-cell">
              ${token.substring(0, 12)}...<br>
              <span class="copy-btn" onclick="copyToClipboard('${shareUrl}')">Copy URL</span>
            </td>
            <td class="${statusClass}">${statusText}</td>
            <td class="${deviceClass}">${deviceBound}</td>
            <td>${info.created}</td>
            <td>${info.first_used || 'Never'}</td>
            <td>${info.last_used || 'Never'}</td>
            <td>${accessCount}</td>
            <td class="${unauthorizedClass}">
              ${unauthorizedCount > 0 ? 
                `<span style="cursor:pointer;" onclick="showUnauthorizedAttempts('${token}')">${unauthorizedCount} ⚠️</span>` : 
                '0'
              }
            </td>
            <td>
              <div class="actions">
                ${info.active ? 
                  `<button class="btn-danger" onclick="revokeToken('${token}')">Revoke</button>` : 
                  `<button class="btn-success" onclick="activateToken('${token}')">Activate</button>`
                }
                ${info.bound_fingerprint ? 
                  `<button onclick="unbindDevice('${token}')">Unbind</button>` : 
                  ''
                }
                <button class="btn-danger" onclick="deleteToken('${token}')">Delete</button>
              </div>
            </td>
          </tr>
        `;
      }
      
      html += '</tbody></table>';
      container.innerHTML = html;
    }
    
    function showUnauthorizedAttempts(token) {
      fetch(ADMIN_API, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: 'get_token_details', token: token })
      })
      .then(response => response.json())
      .then(data => {
        if (data.success && data.token_info.unauthorized_attempts) {
          const attempts = data.token_info.unauthorized_attempts;
          let message = `Unauthorized Access Attempts:\n\n`;
          attempts.forEach((attempt, index) => {
            message += `${index + 1}. Time: ${attempt.timestamp}\n   IP: ${attempt.ip}\n   Fingerprint: ${attempt.fingerprint}\n\n`;
          });
          alert(message);
        }
      });
    }
    
    async function unbindDevice(token) {
      if (!confirm('Are you sure you want to unbind this device? The user will need to access from their device again to rebind.')) return;
      
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ action: 'unbind', token: token })
        });
        
        const data = await response.json();
        if (data.success) {
          alert('Device unbound successfully. User can now bind from a new device.');
          loadTokens();
        } else {
          alert('Error: ' + data.message);
        }
      } catch (error) {
        alert('Error unbinding device');
      }
    }
    
    async function revokeToken(token) {
      if (!confirm('Are you sure you want to revoke this token?')) return;
      
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ action: 'revoke', token: token })
        });
        
        const data = await response.json();
        if (data.success) {
          loadTokens();
        }
      } catch (error) {
        alert('Error revoking token');
      }
    }
    
    async function activateToken(token) {
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ action: 'activate', token: token })
        });
        
        const data = await response.json();
        if (data.success) {
          loadTokens();
        }
      } catch (error) {
        alert('Error activating token');
      }
    }
    
    async function deleteToken(token) {
      if (!confirm('Are you sure you want to permanently delete this token?')) return;
      
      try {
        const response = await fetch(ADMIN_API, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ action: 'delete', token: token })
        });
        
        const data = await response.json();
        if (data.success) {
          loadTokens();
        }
      } catch (error) {
        alert('Error deleting token');
      }
    }
    
    function showResult(elementId, message, type) {
      const element = document.getElementById(elementId);
      element.innerHTML = `<div class="alert alert-${type}">${message}</div>`;
      setTimeout(() => {
        element.innerHTML = '';
      }, 10000);
    }
    
    function copyToClipboard(text) {
      navigator.clipboard.writeText(text).then(() => {
        alert('Copied to clipboard!');
      }).catch(() => {
        prompt('Copy this URL:', text);
      });
    }
    
    // Load tokens on page load
    loadTokens();
  </script>
</body>
</html>