<?php
// proxy.php
header("Content-Type: application/json");

// Get JSON input from frontend
$input = json_decode(file_get_contents("php://input"), true);

if (!isset($input["prompt"])) {
    http_response_code(400);
    echo json_encode(["error" => "Missing prompt"]);
    exit;
}

// API key (better to store in environment variable or config file)
$api_key = getenv("OPENROUTER_API_KEY");
if (!$api_key) {
    $api_key = "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";
}

/** ---------- MODEL & TOKEN LIMITS ---------- **/
$model = 'google/gemma-3-12b-it';
$maxKbTokens = 15000;

// INCREASED max_tokens to allow for 150-200 word responses
// 200 words ≈ 250-300 tokens, adding buffer for safety
$maxResponseTokens = 300;

$ch = curl_init("https://openrouter.ai/api/v1/chat/completions");

$data = [
    "model" => $model,
    "messages" => [
        ["role" => "user", "content" => $input["prompt"]]
    ],
    // CRITICAL CHANGES FOR LONGER RESPONSES:
    "temperature" => 0.1,      // Increased from 0.2 for more expansive responses
    "top_p"       => 0.8,     // Increased from 0.9 for better variety
    "max_tokens"  => $maxResponseTokens,
    "min_tokens"  => 150,      // NEW: Minimum token requirement (150 words ≈ 200 tokens)
    "frequency_penalty" => 0.1, // NEW: Encourages varied vocabulary
    "presence_penalty" => 0.0   // NEW: Encourages exploring new topics
];

curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    "Authorization: Bearer " . $api_key,
    "Content-Type: application/json"
]);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));

$response = curl_exec($ch);

if (curl_errno($ch)) {
    http_response_code(500);
    echo json_encode(["error" => curl_error($ch)]);
    curl_close($ch);
    exit;
}

$http_status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

// Parse the API response to extract the bot's answer
$responseData = json_decode($response, true);
$botAnswer = "";

if (isset($responseData['choices'][0]['message']['content'])) {
    $botAnswer = $responseData['choices'][0]['message']['content'];
    
    // VALIDATION: Check word count
    $wordCount = str_word_count(strip_tags($botAnswer));
    
    // Log word count for debugging
    error_log("Response word count: " . $wordCount);
    
    // Optional: If response is too short, you could flag it or retry
    if ($wordCount < 100) {
        error_log("WARNING: Response is shorter than expected (100 words minimum). Got: " . $wordCount . " words");
    }
}

// Extract only the actual user question (after "Question:" in the prompt)
$actualQuestion = extractUserQuestion($input["prompt"]);

// Log ONLY the question and answer
logConversation($actualQuestion, $botAnswer);

http_response_code($http_status);
echo $response;

/**
 * Extracts the actual user question from the full prompt
 * Looks for "Question:" marker and extracts text after it
 * 
 * @param string $fullPrompt The complete prompt including system instructions
 * @return string The actual user question
 */
function extractUserQuestion($fullPrompt) {
    // Look for "Question:" marker (case-insensitive)
    if (preg_match('/Question:\s*(.+?)(?:\n\n|Give a clear|Provide an authoritative|$)/is', $fullPrompt, $matches)) {
        return trim($matches[1]);
    }
    
    // If no "Question:" marker found, return the full prompt
    // (in case the format changes)
    return trim($fullPrompt);
}

/**
 * Logs only the question and answer to a file on the root server
 * 
 * @param string $question The user's question/prompt
 * @param string $answer The bot's answer
 */
function logConversation($question, $answer) {
    // Define log file path (same directory as proxy.php)
    $logFile = __DIR__ . '/chatbot_log.txt';
    
    // Prepare log entry with word count
    $timestamp = date('Y-m-d H:i:s');
    $separator = str_repeat('=', 80);
    $wordCount = str_word_count(strip_tags($answer));
    
    $logEntry = "\n{$separator}\n";
    $logEntry .= "TIMESTAMP: {$timestamp}\n";
    $logEntry .= "WORD COUNT: {$wordCount}\n";
    $logEntry .= "{$separator}\n";
    $logEntry .= "QUESTION:\n" . trim($question) . "\n\n";
    $logEntry .= "ANSWER:\n" . trim($answer) . "\n";
    $logEntry .= "{$separator}\n";
    
    // Write to log file (append mode)
    $result = file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
    
    // Log any file write errors to PHP error log
    if ($result === false) {
        error_log("Failed to write to chatbot log file: {$logFile}");
    }
    
    // Set file permissions if newly created
    if (file_exists($logFile)) {
        @chmod($logFile, 0600);
    }
}
?>