<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Load the token whitelist
$tokensFile = 'tokens.json';

if (!file_exists($tokensFile)) {
    echo json_encode([
        'valid' => false,
        'message' => 'Token system not initialized. Please contact administrator.'
    ]);
    exit;
}

$tokensData = json_decode(file_get_contents($tokensFile), true);

if (!$tokensData) {
    echo json_encode([
        'valid' => false,
        'message' => 'Error reading token database.'
    ]);
    exit;
}

// Get the token and fingerprint from request
$input = json_decode(file_get_contents('php://input'), true);
$token = $input['token'] ?? '';
$fingerprint = $input['fingerprint'] ?? '';

if (empty($token)) {
    echo json_encode([
        'valid' => false,
        'message' => 'No token provided.'
    ]);
    exit;
}

if (empty($fingerprint)) {
    echo json_encode([
        'valid' => false,
        'message' => 'Device verification failed.'
    ]);
    exit;
}

// Get user's IP address
$userIP = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if (strpos($userIP, ',') !== false) {
    $userIP = trim(explode(',', $userIP)[0]);
}

// Check if token exists
if (!isset($tokensData[$token])) {
    echo json_encode([
        'valid' => false,
        'message' => 'Invalid access token. Contact enquire@kascorner.com'
    ]);
    exit;
}

$tokenInfo = $tokensData[$token];

// Check if token is active
if (!$tokenInfo['active']) {
    echo json_encode([
        'valid' => false,
        'message' => 'This access token has been revoked.'
    ]);
    exit;
}

// Check expiration if set
if (isset($tokenInfo['expires']) && $tokenInfo['expires']) {
    $expiryDate = strtotime($tokenInfo['expires']);
    if ($expiryDate && time() > $expiryDate) {
        echo json_encode([
            'valid' => false,
            'message' => 'This access token has expired.'
        ]);
        exit;
    }
}

// DEVICE BINDING LOGIC
// First time use - bind to this device
if (!isset($tokenInfo['bound_fingerprint']) || empty($tokenInfo['bound_fingerprint'])) {
    $tokensData[$token]['bound_fingerprint'] = $fingerprint;
    $tokensData[$token]['bound_ip'] = $userIP;
    $tokensData[$token]['first_used'] = date('Y-m-d H:i:s');
    $tokensData[$token]['last_used'] = date('Y-m-d H:i:s');
    $tokensData[$token]['access_count'] = 1;
    
    file_put_contents($tokensFile, json_encode($tokensData, JSON_PRETTY_PRINT));
    
    echo json_encode([
        'valid' => true,
        'user' => $tokenInfo['user'] ?? 'Guest',
        'message' => 'Access granted. Token bound to this device.',
        'first_use' => true
    ]);
    exit;
}

// Subsequent use - verify device match
if ($tokenInfo['bound_fingerprint'] !== $fingerprint) {
    // Log unauthorized attempt
    if (!isset($tokensData[$token]['unauthorized_attempts'])) {
        $tokensData[$token]['unauthorized_attempts'] = [];
    }
    
    $tokensData[$token]['unauthorized_attempts'][] = [
        'timestamp' => date('Y-m-d H:i:s'),
        'ip' => $userIP,
        'fingerprint' => substr($fingerprint, 0, 16) . '...'
    ];
    
    file_put_contents($tokensFile, json_encode($tokensData, JSON_PRETTY_PRINT));
    
    echo json_encode([
        'valid' => false,
        'message' => 'This token is registered to another device. Access denied. Contact: enquire@kascorner.com'
    ]);
    exit;
}

// Device matches - update usage stats
$tokensData[$token]['last_used'] = date('Y-m-d H:i:s');
$tokensData[$token]['last_ip'] = $userIP;
$tokensData[$token]['access_count'] = ($tokenInfo['access_count'] ?? 0) + 1;

file_put_contents($tokensFile, json_encode($tokensData, JSON_PRETTY_PRINT));

// Token is valid and device matches
echo json_encode([
    'valid' => true,
    'user' => $tokenInfo['user'] ?? 'Guest',
    'message' => 'Access granted.'
]);
